# Known issues

## Use package [jenssegers/agent](https://github.com/jenssegers/agent)
> [Listen System Event](https://github.com/hhxsv5/laravel-s/blob/master/README.md#system-events)

```PHP
// Reset Agent
\Event::listen('laravels.received_request', function (\Illuminate\Http\Request $req, $app) {
    $app->agent->setHttpHeaders($req->server->all());
    $app->agent->setUserAgent();
});
```

## Use package [barryvdh/laravel-debugbar](https://github.com/barryvdh/laravel-debugbar)
> Not support `cli` mode officially, you need to remove the logic of `runningInConsole`, but there may be some other issues.

```PHP
// Search runningInConsole(), then annotate it
$this->enabled = $configEnabled /*&& !$this->app->runningInConsole()*/ && !$this->app->environment('testing');
```
## Use package [overtrue/wechat](https://github.com/overtrue/wechat)
> The asynchronous notification callback will be failing, because `$app['request']` is empty, give it a value.

```PHP
public function notify(Request $request)
{
    $app = $this->getPayment();//Get payment instance
    $app['request'] = $request;//Add this line to the original code and assign the current request instance to $app['request']
    $response = $app->handlePaidNotify(function ($message, $fail) use($id) {
        //...
    });
    return $response;
}
```
## Use package [laracasts/flash](https://github.com/laracasts/flash)
> Flash messages are held in memory all the time. Appending to `$messages` when call flash() every time, leads to the multiple messages. There are two solutions.

1.Reset `$messages` by middleware `app('flash')->clear();`.

2.Re-register `FlashServiceProvider` after handling request, Refer [register_providers](https://github.com/hhxsv5/laravel-s/blob/master/Settings.md).

## Singleton controller

1.`Incorrect` usage.
```PHP
namespace App\Http\Controllers;
class TestController extends Controller
{
    protected $userId;
    public function __construct()
    {
        // Incorrect usage: TestController is singleton instance, subsequent requests will misread the userId generated by the first request.
        $this->userId = session('userId');
    }
    public function testAction()
    {
        // read $this->userId;
    }
}
```

2.`Correct` usage.
```PHP
namespace App\Http\Controllers;
class TestController extends Controller
{
    protected function getUserId()
    {
        return session('userId');
    }
    public function testAction()
    {
        // call $this->getUserId() to get $userId
    }
}
```

## Cannot call these functions

- `flush`/`ob_flush`/`ob_end_flush`/`ob_implicit_flush`: `swoole_http_response` does not support `flush`.

- `dd()`/`exit()`/`die()`: will lead to Worker/Task/Process quit right now, suggest jump out function call stack by throwing exception.

- `header()`/`setcookie()`/`http_response_code()`: Make HTTP response by Laravel/Lumen `Response` only in LaravelS underlying.

## Cannot use these global variables

- `$_SESSION`

## Size restriction

- The max size of `GET` request's header is `8KB`, restricted by `Swoole`, the big `Cookie` will lead to parse `$_COOKIE` fail.

- The max size of `POST` data/file is restricted by `Swoole` [`package_max_length`](https://www.swoole.co.uk/docs/modules/swoole-server/configuration), default `2M`.

## Inotify reached the watchers limit
> `Warning: inotify_add_watch(): The user limit on the total number of inotify watches was reached`

- Inotify limit is default `8192` for most `Linux`, but the amount of actual project may be more than it, then lead to watch fail.

- Increase the amount of inotify watchers to `524288`: `echo fs.inotify.max_user_watches=524288 | sudo tee -a /etc/sysctl.conf && sudo sysctl -p`, note: you need to enable `privileged` for `Docker`.

## include/require与(include/require)_once
> See Laruence's blog [Do NOT USE (include/require)_once](http://www.laruence.com/2012/09/12/2765.html)

- To include the files about `class`/`interface`/`trait`/`function`, sugguest to use (include/require)_once. In other cases, use include/require.

- In the multi-process mode, the child process inherits the parent process resource. Once the parent process includes a file that needs to be executed, the child process will directly return true when it uses require_once(), causing the file to fail to execute. Now, you need to use include/require.

## If `Swoole < 1.9.17`
> After enabling `handle_static`, static resource files will be handled by `LaravelS`. Due to the PHP environment, `MimeTypeGuesser` may not correctly recognize `MimeType`. For example, Javascript and CSS files will be recognized as `text/plain`.

Solutions:

1.Upgrade Swoole to `1.9.17+`.

2.Register a custom MIME guesser.

```PHP
// MyGuessMimeType.php
use Symfony\Component\HttpFoundation\File\MimeType\MimeTypeGuesserInterface;
class MyGuessMimeType implements MimeTypeGuesserInterface
{
    protected static $map = [
        'js'  => 'application/javascript',
        'css' => 'text/css',
    ];
    public function guess($path)
    {
        $ext = pathinfo($path, PATHINFO_EXTENSION);
        if (strlen($ext) > 0) {
            return Arr::get(self::$map, $ext);
        } else {
            return null;
        }
    }
}
```

```PHP
// AppServiceProvider.php
use Symfony\Component\HttpFoundation\File\MimeType\MimeTypeGuesser;
public function boot()
{
    MimeTypeGuesser::getInstance()->register(new MyGuessMimeType());
}
```

