/*
 * ZGIJKMediaPlayback.h
 *
 * Copyright (c) 2013 Bilibili
 * Copyright (c) 2013 Zhang Rui <bbcallen@gmail.com>
 *
 * This file is part of ijkPlayer.
 *
 * ijkPlayer is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * ijkPlayer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with ijkPlayer; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#import <Foundation/Foundation.h>
#import <UIKit/UIKit.h>

typedef NS_ENUM(NSInteger, IJKMPMovieScalingMode) {
    IJKMPMovieScalingModeNone,       // No scaling
    IJKMPMovieScalingModeAspectFit,  // Uniform scale until one dimension fits
    IJKMPMovieScalingModeAspectFill, // Uniform scale until the movie fills the visible bounds. One dimension may have clipped contents
    IJKMPMovieScalingModeFill        // Non-uniform scale. Both render dimensions will exactly match the visible bounds
};

typedef NS_ENUM(NSInteger, IJKMPMoviePlaybackState) {
    IJKMPMoviePlaybackStateStopped,
    IJKMPMoviePlaybackStatePlaying,
    IJKMPMoviePlaybackStatePaused,
    IJKMPMoviePlaybackStateInterrupted,
    IJKMPMoviePlaybackStateSeekingForward,
    IJKMPMoviePlaybackStateSeekingBackward
};

typedef NS_OPTIONS(NSUInteger, IJKMPMovieLoadState) {
    IJKMPMovieLoadStateUnknown        = 0,
    IJKMPMovieLoadStatePlayable       = 1 << 0,
    IJKMPMovieLoadStatePlaythroughOK  = 1 << 1, // Playback will be automatically started in this state when shouldAutoplay is YES
    IJKMPMovieLoadStateStalled        = 1 << 2, // Playback will be automatically paused in this state, if started
};

typedef NS_ENUM(NSInteger, IJKMPMovieFinishReason) {
    IJKMPMovieFinishReasonPlaybackEnded,
    IJKMPMovieFinishReasonPlaybackError,
    IJKMPMovieFinishReasonUserExited
};

typedef NS_ENUM(NSInteger, IJKMPMoviePullState) {
    IJKMPMoviePullStateFailed,
    IJKMPMoviePullStateSuccessed
};

typedef NS_ENUM(NSInteger, IJKMPMovieWarningType) {
    IJKMPMovieBufferEmptyWarning,
    IJKMPMovieAVLostSynWarning,
    IJKMPMovieNetworkDelayWarning
};

typedef NS_ENUM(NSInteger, IJKMPMovieRequestType) {
    IJKMPMovieTypeLocal,
    IJKMPMovieTypeNetwork
};

typedef NS_ENUM(NSInteger, IJKMPMovieProtocal) {
    IJKMPMovieProtocalRTYMP,
    IJKMPMovieProtocalHLS,
    IJKMPMovieProtocalMP4 = 3
};

typedef NS_ENUM(NSInteger, IJKMPMovieCodecMode) {
    IJKMPMovieUnknownCodec,
    IJKMPMovieMediaCodec,
    IJKMPMovieAVCodec
};

// -----------------------------------------------------------------------------
// Thumbnails

typedef NS_ENUM(NSInteger, IJKMPMovieTimeOption) {
    IJKMPMovieTimeOptionNearestKeyFrame,
    IJKMPMovieTimeOptionExact
};

typedef NS_ENUM(NSInteger, PlayerMode) {
    PLAYER_MODE_LOCAL,
    PLAYER_MODE_VOD,
    PLAYER_MODE_LIVING,
    PLAYER_MODE_P2P
    
};

@protocol ZGIJKMediaPlayback;

#pragma mark ZGIJKMediaPlayback

@protocol ZGIJKMediaPlayback <NSObject>

- (void)prepareToPlay;
- (void)play;
- (void)play_from:(NSTimeInterval)aSeekTime;  //modify 2019-11-11 Deprecated
- (void)pause;
- (void)stop;
- (BOOL)isPlaying;
- (void)shutdown;
- (void)setPauseInBackground:(BOOL)pause;

- (void)setScalingMode: (IJKMPMovieScalingMode) aScalingMode;
- (void)seek:(NSTimeInterval)aSeekTime;
- (NSString*)getVideoCodecInfo;
- (NSString*)getAudioCodecInfo;

- (void)bindPlayerToView:(UIView *)playerView;
- (void)removeView;

- (void)reset;
- (void)mute:(BOOL)mute;
- (void)resetVideoUrl:(NSString*)url;


@property(nonatomic)            NSTimeInterval currentPlaybackTime;
@property(nonatomic, readonly)  NSTimeInterval duration;
@property(nonatomic, readonly)  NSTimeInterval playableDuration;
@property(nonatomic, readonly)  NSInteger bufferingProgress;
//@property(nonatomic, readonly)  double networkBitrate;

@property(nonatomic, readonly)  BOOL isPreparedToPlay;
@property(nonatomic, readonly)  IJKMPMoviePlaybackState playbackState;
@property(nonatomic, readonly)  IJKMPMovieLoadState loadState;

@property(nonatomic, readonly) int64_t numberOfBytesTransferred;

@property(nonatomic, readonly) CGSize naturalSize;
@property(nonatomic) IJKMPMovieScalingMode scalingMode;
@property(nonatomic) BOOL shouldAutoplay;

@property (nonatomic) BOOL allowsMediaAirPlay;
@property (nonatomic) BOOL isDanmakuMediaAirPlay;
@property (nonatomic, readonly) BOOL airPlayMediaActive;

@property (nonatomic) float playbackRate;
@property (nonatomic) float playbackVolume;
@property (nonatomic) PlayerMode playerMode;
@property (nonatomic) NSString* key;

- (UIImage *)thumbnailImageAtCurrentTime;

#pragma mark Notifications

#ifdef __cplusplus
#define IJK_EXTERN extern "C" __attribute__((visibility ("default")))
#else
#define IJK_EXTERN extern __attribute__((visibility ("default")))
#endif

// -----------------------------------------------------------------------------
//  MPMediaPlayback.h

// Posted when the prepared state changes of an object conforming to the MPMediaPlayback protocol changes.
// This supersedes MPMoviePlayerContentPreloadDidFinishNotification.
IJK_EXTERN NSString *const ZGIJKMPMediaPlaybackIsPreparedToPlayDidChangeNotification;

// -----------------------------------------------------------------------------
//  MPMoviePlayerController.h
//  Movie Player Notifications

// Posted when the scaling mode changes.
IJK_EXTERN NSString* const ZGIJKMPMoviePlayerScalingModeDidChangeNotification;

// Posted when movie playback ends or a user exits playback.
IJK_EXTERN NSString* const ZGIJKMPMoviePlayerPlaybackDidFinishNotification;
IJK_EXTERN NSString* const ZGIJKMPMoviePlayerPlaybackDidFinishReasonUserInfoKey; // NSNumber (IJKMPMovieFinishReason)

// Posted when the playback state changes, either programatically or by the user.
IJK_EXTERN NSString* const ZGIJKMPMoviePlayerPlaybackStateDidChangeNotification;

// Posted when the network load state changes.
IJK_EXTERN NSString* const ZGIJKMPMoviePlayerLoadStateDidChangeNotification;

// Posted when the movie player begins or ends playing video via AirPlay.
IJK_EXTERN NSString* const ZGIJKMPMoviePlayerIsAirPlayVideoActiveDidChangeNotification;

// -----------------------------------------------------------------------------
// Movie Property Notifications

// Calling -prepareToPlay on the movie player will begin determining movie properties asynchronously.
// These notifications are posted when the associated movie property becomes available.
IJK_EXTERN NSString* const ZGIJKMPMovieNaturalSizeAvailableNotification;

// -----------------------------------------------------------------------------
//  Extend Notifications

IJK_EXTERN NSString *const ZGIJKMPMoviePlayerVideoDecoderOpenNotification;
IJK_EXTERN NSString *const ZGIJKMPMoviePlayerFirstVideoFrameRenderedNotification;
IJK_EXTERN NSString *const ZGIJKMPMoviePlayerFirstAudioFrameRenderedNotification;

IJK_EXTERN NSString *const ZGIJKMPMoviePlayerDidSeekCompleteNotification;
IJK_EXTERN NSString *const ZGIJKMPMoviePlayerDidSeekCompleteTargetKey;
IJK_EXTERN NSString *const ZGIJKMPMoviePlayerDidSeekCompleteErrorKey;
IJK_EXTERN NSString *const ZGIJKMPMoviePlayerDidAccurateSeekCompleteCurPos;
IJK_EXTERN NSString *const ZGIJKMPMoviePlayerAccurateSeekCompleteNotification;

IJK_EXTERN NSString *const ZGIJKMPMoviePlayerDidKeyFrameNotification;

IJK_EXTERN NSString *const ZGIJKMPMoviePlayerDidBufferEnoughNotification;

IJK_EXTERN NSString *const ZGIJKMPMoviePlayerDidPullStreamStateNotification;
IJK_EXTERN NSString *const ZGIJKMPMoviePlayerDidPullStreamStateUserInfoKey;

IJK_EXTERN NSString *const ZGIJKMPMoviePlayerDidKeyFrameNotification;

IJK_EXTERN NSString *const ZGIJKMPMoviePlayerWarningNotification;
IJK_EXTERN NSString *const ZGIJKMPMoviePlayerWarningUserInfoKey;

IJK_EXTERN NSString *const ZGIJKMPMoviePlayerLoadTimeRanges;
@end

#pragma mark IJKMediaUrlOpenDelegate

// Must equal to the defination in ijkavformat/ijkavformat.h
typedef NS_ENUM(NSInteger, IJKMediaEvent) {

    // Notify Events
    IJKMediaEvent_WillHttpOpen         = 1,       // attr: url
    IJKMediaEvent_DidHttpOpen          = 2,       // attr: url, error, http_code
    IJKMediaEvent_WillHttpSeek         = 3,       // attr: url, offset
    IJKMediaEvent_DidHttpSeek          = 4,       // attr: url, offset, error, http_code
    // Control Message
    IJKMediaCtrl_WillTcpOpen           = 0x20001, // ZGIJKMediaUrlOpenData: no args
    IJKMediaCtrl_DidTcpOpen            = 0x20002, // ZGIJKMediaUrlOpenData: error, family, ip, port, fd
    IJKMediaCtrl_WillHttpOpen          = 0x20003, // ZGIJKMediaUrlOpenData: url, segmentIndex, retryCounter
    IJKMediaCtrl_WillLiveOpen          = 0x20005, // ZGIJKMediaUrlOpenData: url, retryCounter
    IJKMediaCtrl_WillConcatSegmentOpen = 0x20007, // ZGIJKMediaUrlOpenData: url, segmentIndex, retryCounter
};

#define IJKMediaEventAttrKey_url            @"url"
#define IJKMediaEventAttrKey_host           @"host"
#define IJKMediaEventAttrKey_error          @"error"
#define IJKMediaEventAttrKey_time_of_event  @"time_of_event"
#define IJKMediaEventAttrKey_http_code      @"http_code"
#define IJKMediaEventAttrKey_offset         @"offset"

// event of IJKMediaUrlOpenEvent_xxx
@interface ZGIJKMediaUrlOpenData: NSObject

- (id)initWithUrl:(NSString *)url
            event:(IJKMediaEvent)event
     segmentIndex:(int)segmentIndex
     retryCounter:(int)retryCounter;

@property(nonatomic, readonly) IJKMediaEvent event;
@property(nonatomic, readonly) int segmentIndex;
@property(nonatomic, readonly) int retryCounter;

@property(nonatomic, retain) NSString *url;
@property(nonatomic, assign) int fd;
@property(nonatomic, strong) NSString *msg;
@property(nonatomic) int error; // set a negative value to indicate an error has occured.
@property(nonatomic, getter=isHandled)    BOOL handled;     // auto set to YES if url changed
@property(nonatomic, getter=isUrlChanged) BOOL urlChanged;  // auto set to YES by url changed

@end

@protocol IJKMediaUrlOpenDelegate <NSObject>

- (void)willOpenUrl:(ZGIJKMediaUrlOpenData*) urlOpenData;

@end

@protocol IJKMediaNativeInvokeDelegate <NSObject>

- (int)invoke:(IJKMediaEvent)event attributes:(NSDictionary *)attributes;

@end

