/*
 * This file is part of the ZYJSDWebImage package.
 * (c) Olivier Poitrey <rs@dailymotion.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

#import "ZYJSDWebImageCompat.h"

#if ZYJSD_UIKIT

#import "ZYJSDWebImageManager.h"

/**
 * Integrates ZYJSDWebImage async downloading and caching of remote images with UIImageView for highlighted state.
 */
@interface UIImageView (HighlightedZYJWebCache)

/**
 * Set the imageView `highlightedImage` with an `url`.
 *
 * The download is asynchronous and cached.
 *
 * @param url The url for the image.
 */
- (void)ZYJSD_setHighlightedImageWithURL:(nullable NSURL *)url NS_REFINED_FOR_SWIFT;

/**
 * Set the imageView `highlightedImage` with an `url` and custom options.
 *
 * The download is asynchronous and cached.
 *
 * @param url     The url for the image.
 * @param options The options to use when downloading the image. @see ZYJSDWebImageOptions for the possible values.
 */
- (void)ZYJSD_setHighlightedImageWithURL:(nullable NSURL *)url
                              options:(ZYJSDWebImageOptions)options NS_REFINED_FOR_SWIFT;

/**
 * Set the imageView `highlightedImage` with an `url`, custom options and context.
 *
 * The download is asynchronous and cached.
 *
 * @param url     The url for the image.
 * @param options The options to use when downloading the image. @see ZYJSDWebImageOptions for the possible values.
 * @param context     A context contains different options to perform specify changes or processes, see `ZYJSDWebImageContextOption`. This hold the extra objects which `options` enum can not hold.
 */
- (void)ZYJSD_setHighlightedImageWithURL:(nullable NSURL *)url
                              options:(ZYJSDWebImageOptions)options
                              context:(nullable ZYJSDWebImageContext *)context;

/**
 * Set the imageView `highlightedImage` with an `url`.
 *
 * The download is asynchronous and cached.
 *
 * @param url            The url for the image.
 * @param completedBlock A block called when operation has been completed. This block has no return value
 *                       and takes the requested UIImage as first parameter. In case of error the image parameter
 *                       is nil and the second parameter may contain an NSError. The third parameter is a Boolean
 *                       indicating if the image was retrieved from the local cache or from the network.
 *                       The fourth parameter is the original image url.
 */
- (void)ZYJSD_setHighlightedImageWithURL:(nullable NSURL *)url
                            completed:(nullable ZYJSDExternalCompletionBlock)completedBlock NS_REFINED_FOR_SWIFT;

/**
 * Set the imageView `highlightedImage` with an `url` and custom options.
 *
 * The download is asynchronous and cached.
 *
 * @param url            The url for the image.
 * @param options        The options to use when downloading the image. @see ZYJSDWebImageOptions for the possible values.
 * @param completedBlock A block called when operation has been completed. This block has no return value
 *                       and takes the requested UIImage as first parameter. In case of error the image parameter
 *                       is nil and the second parameter may contain an NSError. The third parameter is a Boolean
 *                       indicating if the image was retrieved from the local cache or from the network.
 *                       The fourth parameter is the original image url.
 */
- (void)ZYJSD_setHighlightedImageWithURL:(nullable NSURL *)url
                              options:(ZYJSDWebImageOptions)options
                            completed:(nullable ZYJSDExternalCompletionBlock)completedBlock;

/**
 * Set the imageView `highlightedImage` with an `url` and custom options.
 *
 * The download is asynchronous and cached.
 *
 * @param url            The url for the image.
 * @param options        The options to use when downloading the image. @see ZYJSDWebImageOptions for the possible values.
 * @param progressBlock  A block called while image is downloading
 *                       @note the progress block is executed on a background queue
 * @param completedBlock A block called when operation has been completed. This block has no return value
 *                       and takes the requested UIImage as first parameter. In case of error the image parameter
 *                       is nil and the second parameter may contain an NSError. The third parameter is a Boolean
 *                       indicating if the image was retrieved from the local cache or from the network.
 *                       The fourth parameter is the original image url.
 */
- (void)ZYJSD_setHighlightedImageWithURL:(nullable NSURL *)url
                              options:(ZYJSDWebImageOptions)options
                             progress:(nullable ZYJSDImageLoaderProgressBlock)progressBlock
                            completed:(nullable ZYJSDExternalCompletionBlock)completedBlock;

/**
 * Set the imageView `highlightedImage` with an `url`, custom options and context.
 *
 * The download is asynchronous and cached.
 *
 * @param url            The url for the image.
 * @param options        The options to use when downloading the image. @see ZYJSDWebImageOptions for the possible values.
 * @param context     A context contains different options to perform specify changes or processes, see `ZYJSDWebImageContextOption`. This hold the extra objects which `options` enum can not hold.
 * @param progressBlock  A block called while image is downloading
 *                       @note the progress block is executed on a background queue
 * @param completedBlock A block called when operation has been completed. This block has no return value
 *                       and takes the requested UIImage as first parameter. In case of error the image parameter
 *                       is nil and the second parameter may contain an NSError. The third parameter is a Boolean
 *                       indicating if the image was retrieved from the local cache or from the network.
 *                       The fourth parameter is the original image url.
 */
- (void)ZYJSD_setHighlightedImageWithURL:(nullable NSURL *)url
                              options:(ZYJSDWebImageOptions)options
                              context:(nullable ZYJSDWebImageContext *)context
                             progress:(nullable ZYJSDImageLoaderProgressBlock)progressBlock
                            completed:(nullable ZYJSDExternalCompletionBlock)completedBlock;

@end

#endif
