/*
 * This file is part of the ZYJSDWebImage package.
 * (c) Olivier Poitrey <rs@dailymotion.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

#import <Foundation/Foundation.h>
#import "ZYJSDWebImageCompat.h"
#import "ZYJSDWebImageDefine.h"

@class ZYJSDWebImageOptionsResult;

typedef ZYJSDWebImageOptionsResult * _Nullable(^ZYJSDWebImageOptionsProcessorBlock)(NSURL * _Nullable url, ZYJSDWebImageOptions options, ZYJSDWebImageContext * _Nullable context);

/**
 The options result contains both options and context.
 */
@interface ZYJSDWebImageOptionsResult : NSObject

/**
 ZYJWebCache options.
 */
@property (nonatomic, assign, readonly) ZYJSDWebImageOptions options;

/**
 Context options.
 */
@property (nonatomic, copy, readonly, nullable) ZYJSDWebImageContext *context;

/**
 Create a new options result.

 @param options options
 @param context context
 @return The options result contains both options and context.
 */
- (nonnull instancetype)initWithOptions:(ZYJSDWebImageOptions)options context:(nullable ZYJSDWebImageContext *)context;

@end

/**
 This is the protocol for options processor.
 Options processor can be used, to control the final result for individual image request's `ZYJSDWebImageOptions` and `ZYJSDWebImageContext`
 Implements the protocol to have a global control for each indivadual image request's option.
 */
@protocol ZYJSDWebImageOptionsProcessor <NSObject>

/**
 Return the processed options result for specify image URL, with its options and context

 @param url The URL to the image
 @param options A mask to specify options to use for this request
 @param context A context contains different options to perform specify changes or processes, see `ZYJSDWebImageContextOption`. This hold the extra objects which `options` enum can not hold.
 @return The processed result, contains both options and context
 */
- (nullable ZYJSDWebImageOptionsResult *)processedResultForURL:(nullable NSURL *)url
                                                    options:(ZYJSDWebImageOptions)options
                                                    context:(nullable ZYJSDWebImageContext *)context;

@end

/**
 A options processor class with block.
 */
@interface ZYJSDWebImageOptionsProcessor : NSObject<ZYJSDWebImageOptionsProcessor>

- (nonnull instancetype)initWithBlock:(nonnull ZYJSDWebImageOptionsProcessorBlock)block;
+ (nonnull instancetype)optionsProcessorWithBlock:(nonnull ZYJSDWebImageOptionsProcessorBlock)block;

@end
