<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;
use App\Http\Services\CourseService;

class CourseController extends BaseController
{
    //数据处理层初始化
    protected $courseService;

    /**
     * 构造方法,初始化service
     * @access public
     * @param Request $request Request 对象
     */
    public function __construct() {
        $this->courseService = new CourseService();
    }

    /**
     * 添加课程
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addCourse(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $courseInfo = $request->json()->all();

        $checkResult = $this->courseService->checkAddCourseHandle($courseInfo);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->courseService->addCourseHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 获取课程列表
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCourseList(Request $request) {
        $userInfo = $request->attributes->get('user_info');

        $searchParam = [];
        $searchParam['name'] = $request->input('name', '');
        $searchParam['category_id'] = $request->input('category_id', '');
        $searchParam['status'] = $request->input('status', '');
        $searchParam['is_recommend'] = $request->input('is_recommend', '');
        $searchParam['course_code'] = $request->input('course_code', '');
        $searchParam['page'] = $request->input('page', 1);
        $searchParam['page_size'] = $request->input('page_size', 20);

        $checkResult = $this->courseService->checkCourseListSearchHandle($searchParam);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }
        $checkResult['data']['platform'] = $request->input('platform', '');

        $result = $this->courseService->getCourseListHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 修改课程发布状态
     * @param Request $request
     */
    public function updateCourseStatus(Request $request) {
        $courseInfo = $request->json()->all();

        $result = $this->courseService->updateCourseStatusHandle($courseInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 删除课程(支持批量)
     * @param Request $request
     */
    public function deleteCourse(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $courseIds = $request->json()->all();

        $result = $this->courseService->deleteCourseHandle($userInfo, $courseIds);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 获取课程详情
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCourseInfo(Request $request) {
        $courseId = $request->input('course_id', 0);

        $result = $this->courseService->getCourseInfoHandle($courseId);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 更新课程
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateCourse(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $courseInfo = $request->json()->all();

        $checkResult = $this->courseService->checkUpdateCourseHandle($courseInfo);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->courseService->updateCourseHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 推荐课程
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function recommendCourse(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $courseInfo = $request->json()->all();

        $result = $this->courseService->recommendCourseHandle($userInfo, $courseInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 置顶课程
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function setCourseTop(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $courseInfo = $request->json()->all();

        $result = $this->courseService->setCourseTopHandle($userInfo, $courseInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 获取课程实时库存
     * @param Request $request
     */
    public function getCourseStock(Request $request) {
        $courseIds = $request->input('course_ids', '');

        $result = $this->courseService->getCourseStockHandle($courseIds);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 修改课程销量
     * @param Request $request
     */
    public function updateCourseSales(Request $request) {
        $courseInfo = $request->json()->all();

        $result = $this->courseService->updateCourseSalesHandle($courseInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 课程评价
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addCourseEvaluate(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $courseInfo = $request->json()->all();

        $checkResult = $this->courseService->checkAddCourseEvaluate($courseInfo);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->courseService->addCourseEvaluateHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 获取课程评价列表
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCourseEvaluateList(Request $request) {
        $userInfo = $request->attributes->get('user_info');

        $searchParam = [];
        $searchParam['course_id'] = $request->input('course_id', '');
        $searchParam['page'] = $request->input('page', 1);
        $searchParam['page_size'] = $request->input('page_size', 20);

        $checkResult = $this->courseService->checkCourseEvaluateListSearchHandle($searchParam);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->courseService->getCourseEvaluateListHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 后台获取课程评价列表
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCourseEvaluateAdminList(Request $request) {
        $userInfo = $request->attributes->get('user_info');

        $searchParam = [];
        $searchParam['course_id'] = $request->input('course_id', '');
        $searchParam['status'] = $request->input('status', '');
        $searchParam['date_start'] = $request->input('date_start', '');
        $searchParam['date_end'] = $request->input('date_end', '');
        $searchParam['page'] = $request->input('page', 1);
        $searchParam['page_size'] = $request->input('page_size', 20);

        $checkResult = $this->courseService->checkCourseEvaluateAdminListSearchHandle($searchParam);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->courseService->getCourseEvaluateAdminListHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 后台获取课程评价列表
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCourseEvaluateDetail(Request $request) {
        $userInfo = $request->attributes->get('user_info');

        $searchParam = [];
        $searchParam['evaluate_id'] = (int)$request->input('evaluate_id', '');

        $result = $this->courseService->getCourseEvaluateDetailHandle($userInfo, $searchParam);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 评价审核
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function checkCourseEvaluate(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $checkInfo = $request->json()->all();

        $result = $this->courseService->checkCourseEvaluateHandle($userInfo, $checkInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

}
