<?php

namespace App\Http\Controllers;
use App\Http\Services\BaseService;
use Illuminate\Http\Request;
use App\Http\Services\OrderService;
use Illuminate\Support\Facades\Storage;

class OrderController extends BaseController
{
    //数据处理层初始化
    protected $orderService;

    /**
     * 构造方法,初始化service
     * @access public
     * @param Request $request Request 对象
     */
    public function __construct() {
        $this->orderService = new OrderService();
    }

    /**
     * 提交订单
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addOrder(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $courseInfo = $request->json()->all();

        $result = $this->orderService->addOrderHandle($userInfo, $courseInfo);
        if (empty($result['res'])) {
            return $this->respondError($result['code'] ?? 30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 获取订单课程列表
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getOrderCourseList(Request $request) {
        $userInfo = $request->attributes->get('user_info');

        $searchParam = [];
        $searchParam['page'] = $request->input('page', 1);
        $searchParam['page_size'] = $request->input('page_size', 20);

        $result = $this->orderService->getOrderCourseListHandle($userInfo, $searchParam);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 获取订单列表
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getOrderList(Request $request) {
        $userInfo = $request->attributes->get('user_info');

        $searchParam = [];
        $searchParam['name'] = $request->input('name', '');
        $searchParam['phone'] = $request->input('phone', '');
        $searchParam['class_phone'] = $request->input('class_phone', '');
        $searchParam['province_name'] = $request->input('province_name', '');
        $searchParam['date_start'] = $request->input('date_start', '');
        $searchParam['date_end'] = $request->input('date_end', '');
        $searchParam['status'] = $request->input('status', '');
        $searchParam['reception_name'] = $request->input('reception_name', '');
        $searchParam['is_export'] = $request->input('is_export', 0);
        $searchParam['is_finished'] = $request->input('is_finished', '');
        $searchParam['pay_type'] = $request->input('pay_type', '');
        $searchParam['order_type'] = $request->input('order_type', '');
        $searchParam['page'] = $request->input('page', 1);
        $searchParam['page_size'] = $request->input('page_size', 20);
       
        // 根据 用地区 ，权限处理
        if($userInfo['is_admin'] != 1 && strpos(','.$userInfo['area_pro_name'].',', '全国' ) === false) {
            if(strlen($searchParam['province_name']) == 0) {
                // 只处理 当前用户支持 的省份
                $searchParam['area_pro_name'] = $userInfo['area_pro_name'];
            }else {
                // 查询 省份，是否 是当前用户支持的省份
                if(strpos(','.$userInfo['area_pro_name'].',', ','.$searchParam['province_name'].',' ) === false){
                    return $this->respondError(30005,  '无权限查看该省份订单');
                }else{
                    $searchParam['area_pro_name'] = $userInfo['area_pro_name']; 
                }
            }
            if(strlen($userInfo['area_pro_name']) == 0) {
                return $this->respondError(30006,  '无权限查看该订单：请联系管理员配置账号地区权限');
            }
        }else{
            // 管理员 ，可以处理所有省份
            $searchParam['area_pro_name']  = '';
        }

        $checkResult = $this->orderService->checkOrderListSearchHandle($searchParam);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->orderService->getOrderListHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 查看订单详情
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getOrderInfo(Request $request) {
        $orderId = $request->input('order_id', 0);

        $result = $this->orderService->getOrderInfoHandle($orderId);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 更新订单物流信息
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateOrderExpress(Request $request) {
        $expressInfo = $request->json()->all();
        $userInfo = $request->attributes->get('user_info');

        $checkResult = $this->orderService->checkUpdateOrderExpressHandle($expressInfo);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->orderService->updateOrderExpressHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 查看我的快递
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getExpressList(Request $request) {
        $userInfo = $request->attributes->get('user_info');

        $result = $this->orderService->getExpressListHandle($userInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 更新接待人信息
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateOrderReception(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $receptionInfo = $request->json()->all();

        $checkResult = $this->orderService->checkUpdateOrderReceptionHandle($receptionInfo);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->orderService->updateOrderReceptionHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 驳回订单
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function orderReject(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $orderInfo = $request->json()->all();

        $result = $this->orderService->orderRejectHandle($userInfo, $orderInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 上传协议
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function uploadProtocol(Request $request) {
        $protocolFile = $request->file('file');
        $platform = $request->input('platform');
        $orderId = (int)$request->input('order_id', '');
        $userInfo = $request->attributes->get('user_info');

        $result = $this->orderService->uploadProtocolHandle($platform, $protocolFile, $orderId, $userInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 撤回订单
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function orderCancel(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $orderInfo = $request->json()->all();

        $result = $this->orderService->orderCancelHandle($userInfo, $orderInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 获取我的协议
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProtocolList(Request $request) {
        $userInfo = $request->attributes->get('user_info');

        $result = $this->orderService->getProtocolListHandle($userInfo);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 推送协议
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function orderProtocolPush(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $orderInfo = $request->json()->all();

        $result = $this->orderService->orderProtocolPushHandle($userInfo, $orderInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 协议审核操作
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function orderProtocolCheck(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $orderInfo = $request->json()->all();

        $result = $this->orderService->orderProtocolCheckHandle($userInfo, $orderInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 订单删除,支持批量
     * @param Request $request
     */
    public function deleteOrders(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $orderInfo = $request->json()->all();

        $result = $this->orderService->deleteOrdersHandle($userInfo, $orderInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 订单类型修改
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function orderRelease(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $orderInfo = $request->json()->all();

        $result = $this->orderService->orderReleaseHandle($userInfo, $orderInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 修改订单沟通情况
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function orderCommunication(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $orderInfo = $request->json()->all();

        $result = $this->orderService->orderCommunicationHandle($userInfo, $orderInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }


    /**
     * 新增交易记录
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addDeal(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $dealInfo = $request->json()->all();

        $checkResult = $this->orderService->checkAddDealHandle($dealInfo);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->orderService->addDealHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 获取交易记录列表
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDealList(Request $request) {
        $userInfo = $request->attributes->get('user_info');

        $searchParam = [];
        $searchParam['province_name'] = $request->input('province_name', '');
        $searchParam['reception_name'] = $request->input('reception_name', '');
        $searchParam['user_name'] = $request->input('user_name', '');
        $searchParam['class_phone'] = $request->input('class_phone', '');
        $searchParam['department'] = $request->input('department', '');
        $searchParam['date_start'] = $request->input('date_start', '');
        $searchParam['date_end'] = $request->input('date_end', '');
        $searchParam['status'] = $request->input('status', '');
        $searchParam['is_set_express'] = $request->input('is_set_express', '');
        $searchParam['goods_name'] = $request->input('goods_name', '');
        $searchParam['system_status'] = $request->input('system_status', '');
        $searchParam['deal_record_no'] = $request->input('deal_record_no', '');
        $searchParam['deliver_status'] = $request->input('deliver_status', '');
        $searchParam['order_no'] = $request->input('order_no', '');
        $searchParam['is_court'] = $request->input('is_court', '');
        $searchParam['check_status'] = $request->input('check_status', '');
        $searchParam['export_type'] = $request->input('export_type', 0);
        $searchParam['page'] = $request->input('page', 1);
        $searchParam['page_size'] = $request->input('page_size', 20);

        // 判断省份, 管理员和全国, 可以看所有省份
        if (!empty($userInfo['is_admin']) || strpos(','.$userInfo['deal_area_pro_name'].',', '全国' ) !== false) {
            $searchParam['deal_area_pro_name']  = '';
        } else {
            // 判断特殊的总部账号, 这些账号只能看自己的
            $configValue = (new BaseService())->getConfigValue('zongbu_account');
            if (empty($configValue)) {
                return $this->respondError(30007, '请联系管理员配置账号信息');
            }

            $zongbuAccount = explode(',', $configValue);
            // 如果当前账号是在这个名单里,就只能看自己的,不区分省份
            if (in_array($userInfo['phone'], $zongbuAccount)) {
                $searchParam['created_by'] = $userInfo['id'];
            } else {
                // 其余的需要判断省份权限
                if (empty($userInfo['deal_area_pro_name'])) {
                    return $this->respondError(30006,  '无权限查看: 请联系管理员配置账号地区权限');
                }

                // 非总部的, 判断省份权限
                if ($searchParam['province_name'] === '') {
                    $searchParam['deal_area_pro_name'] = $userInfo['deal_area_pro_name'];
                } else if (strpos(','.$userInfo['deal_area_pro_name'].',', ','.$searchParam['province_name'].',' ) === false) {
                    // 搜索了省份, 判断省份是否在权限中
                    return $this->respondError(30005,  '无权限查看该省份订单');
                }
            }
        }

        $checkResult = $this->orderService->checkDealListSearchHandle($searchParam);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->orderService->getDealListHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 获取交易记录详情
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDealInfo(Request $request) {
        $dealRecordId = $request->input('deal_record_id', 0);

        $result = $this->orderService->getDealInfoHandle($dealRecordId);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 编辑保存交易记录
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateDeal(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $dealInfo = $request->json()->all();

        $checkResult = $this->orderService->checkUpdateDealHandle($dealInfo);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->orderService->updateDealHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 删除交易记录(支持批量)
     * @param Request $request
     */
    public function deleteDeal(Request $request) {
        $dealRecordInfo = $request->json()->all();
        $userInfo = $request->attributes->get('user_info');

        $result = $this->orderService->deleteDealHandle($userInfo, $dealRecordInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 设置交易记录无效(支持批量)
     * @param Request $request
     */
    public function setDealInvalid(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $dealRecordInfo = $request->json()->all();

        $result = $this->orderService->setDealInvalidHandle($userInfo, $dealRecordInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 设置系统状态(支持批量)
     * @param Request $request
     */
    public function setDealSystemStatus(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $dealRecordInfo = $request->json()->all();

        $result = $this->orderService->setDealSystemStatusHandle($userInfo, $dealRecordInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 提交审核(支持批量)
     * @param Request $request
     */
    public function dealSubmitCheck(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $dealRecordInfo = $request->json()->all();

        $result = $this->orderService->dealSubmitCheckHandle($userInfo, $dealRecordInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 标记状态导入
     * @param Request $request
     */
    public function importSystemStatus(Request $request) {
        $file = $request->file('file');
        $userInfo = $request->attributes->get('user_info');

        $result = $this->orderService->importSystemStatusHandle($file, $userInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 15天内未发货商品列表
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getGoodsNotDeliverList(Request $request) {
        $searchParam = [];
        $searchParam['user_name'] = $request->input('user_name', '');
        $searchParam['class_phone'] = $request->input('class_phone', '');
        $searchParam['goods_name'] = $request->input('goods_name', '');
        $searchParam['deal_record_no'] = $request->input('deal_record_no', '');
        $searchParam['reception_name'] = $request->input('reception_name', '');
        $searchParam['department'] = $request->input('department', '');
        $searchParam['province_name'] = $request->input('province_name', '');
        $searchParam['order_no'] = $request->input('order_no', '');
        $searchParam['page'] = $request->input('page', 1);
        $searchParam['page_size'] = $request->input('page_size', 20);

        $checkResult = $this->orderService->checkGoodsNotDeliverListSearchHandle($searchParam);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->orderService->getGoodsNotDeliverListHandle($checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 交易记录审核/驳回(支持批量)
     * @param Request $request
     */
    public function checkDeal(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $dealRecordInfo = $request->json()->all();

        $result = $this->orderService->checkDealHandle($userInfo, $dealRecordInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 交易记录导入
     * @param Request $request
     */
    public function importDeal(Request $request) {
        $file = $request->file('file');
        $userInfo = $request->attributes->get('user_info');

        $result = $this->orderService->importDealHandle($file, $userInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 校验协议单号
     * @param Request $request
     */
    public function checkProtocolOrder(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $dealOrderInfo = $request->json()->all();

        $result = $this->orderService->checkProtocolOrderHandle($userInfo, $dealOrderInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '', $result['data']);
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 更新交易记录的协议
     * @param Request $request
     */
    public function updateDealRecordProtocol(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $dealOrderInfo = $request->json()->all();

        $result = $this->orderService->updateDealRecordProtocolHandle($userInfo, $dealOrderInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '', $result['data']);
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 快递单号导入
     * @param Request $request
     */
    public function importDealExpress(Request $request) {
        $file = $request->file('file');
        $userInfo = $request->attributes->get('user_info');

        $result = $this->orderService->importDealExpressHandle($file, $userInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 更新商品快递单号
     * @param Request $request
     */
    public function updateDealExpress(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $dealRecordInfo = $request->json()->all();

        $result = $this->orderService->updateDealExpressHandle($userInfo, $dealRecordInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 新增商品
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addGoods(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        $goodsInfo = $request->json()->all();

        $checkResult = $this->orderService->checkAddGoodsHandle($goodsInfo);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->orderService->addGoodsHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 获取商品列表
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getGoodsList(Request $request) {
        $searchParam = [];
        $searchParam['province_name'] = $request->input('province_name', '');
        $searchParam['category_name'] = $request->input('category_name', '');
        $searchParam['supplier_name'] = $request->input('supplier_name', '');
        $searchParam['model_name'] = $request->input('model_name', '');
        $searchParam['goods_name'] = $request->input('goods_name', '');
        $searchParam['status'] = $request->input('status', '');
        $searchParam['is_export'] = $request->input('is_export', 0);
        $searchParam['page'] = $request->input('page', 1);
        $searchParam['page_size'] = $request->input('page_size', 20);

        $checkResult = $this->orderService->checkGoodsListSearchHandle($searchParam);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->orderService->getGoodsListHandle($checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30002, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 商品禁用/启用
     * @param Request $request
     */
    public function updateGoodsStatus(Request $request) {
        $goodsInfo = $request->json()->all();

        $result = $this->orderService->updateGoodsStatusHandle($goodsInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 商品导入
     * @param Request $request
     */
    public function importGoods(Request $request) {
        $file = $request->file('file');
        $userInfo = $request->attributes->get('user_info');

        $result = $this->orderService->importGoodsHandle($file, $userInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 编辑商品成本价
     * @param Request $request
     */
    public function editGoodsPrice(Request $request) {
        $goodsInfo = $request->json()->all();
        $userInfo = $request->attributes->get('user_info');

        $result = $this->orderService->editGoodsPriceHandle($goodsInfo, $userInfo);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 商品关联小程序/修改售价
     * @param Request $request
     */
    public function goodsRelatedApplet(Request $request) {
        $goodsInfo = $request->json()->all();
        $userInfo = $request->attributes->get('user_info');

        $checkResult = $this->orderService->checkGoodsRelatedAppletHandle($goodsInfo);
        if (empty($checkResult['res'])) {
            return $this->respondError(30001, $checkResult['msg'] ?? '');
        }

        $result = $this->orderService->goodsRelatedAppletHandle($userInfo, $checkResult['data']);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 成交情况汇总
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDealStatistics(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        if (empty($userInfo['is_admin'])) {
            return $this->respondError(30007, '您无权限查看,请联系管理员');
        }

        $searchParam['date_start'] = $request->input('date_start', '');
        $searchParam['date_end'] = $request->input('date_end', '');

        $result = $this->orderService->getDealStatisticsHandle($searchParam);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 商品分类统计
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDealGoodsStatistics(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        if (empty($userInfo['is_admin'])) {
            return $this->respondError(30007, '您无权限查看,请联系管理员');
        }

        $searchParam['date_start'] = $request->input('date_start', '');
        $searchParam['date_end'] = $request->input('date_end', '');

        $result = $this->orderService->getDealGoodsStatisticsHandle($searchParam);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 分校成交情况汇总
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDealBranchStatistics(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        if (empty($userInfo['is_admin'])) {
            return $this->respondError(30007, '您无权限查看,请联系管理员');
        }

        $searchParam['month'] = $request->input('month', '');

        $result = $this->orderService->getDealBranchStatisticsHandle($searchParam);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }

    /**
     * 商品分类分部门统计
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getGoodsDepartmentStatistics(Request $request) {
        $userInfo = $request->attributes->get('user_info');
        if (empty($userInfo['is_admin'])) {
            return $this->respondError(30007, '您无权限查看,请联系管理员');
        }

        $searchParam['month'] = $request->input('month', '');
        $searchParam['province_name'] = $request->input('province_name', '');

        $result = $this->orderService->getGoodsDepartmentStatisticsHandle($searchParam);
        if (empty($result['res'])) {
            return $this->respondError(30001, $result['msg'] ?? '');
        }

        return $this->respondSuccess($result['data'] ?? []);
    }


}
