#include "cmd_processer_client.h"
#include "mqtt_request.h"

#include <time.h>

namespace offcn
{
	static const std::string kGlobalTopic = "kcp_server_mqtt";
	static const std::string kLocalID = "mqtt_client";
	static const std::string kDstTopic = kLocalID;

	static std::string clientMessageID()
	{
		char str[256] = { 0 };
		int len = 12;

		srand((unsigned int)time(NULL));
		int i;
		for (i = 0; i < len; ++i)
		{
			switch ((rand() % 3))
			{
			case 1:
				str[i] = 'A' + rand() % 26;
				break;
			case 2:
				str[i] = 'a' + rand() % 26;
				break;
			default:
				str[i] = '0' + rand() % 10;
				break;
			}
		}
		str[++i] = '\0';
		return str;
	}

	CmdProcesserClient::CmdProcesserClient()
		:mqtt_wrapper_(NULL),
		observer_(NULL)
	{
		local_id_ = kLocalID;

		mqtt_wrapper_ = new offcn::MqttWrapper(this);
	}
	CmdProcesserClient::~CmdProcesserClient()
	{
		if (mqtt_wrapper_)
		{
			delete mqtt_wrapper_;
			mqtt_wrapper_ = NULL;
		}
	}

	void CmdProcesserClient::SetLocalID(std::string id, CmdProcesserObserver *observer)
	{
		local_id_ = id;
		observer_ = observer;
	}
	bool CmdProcesserClient::Connect()
	{
		return mqtt_wrapper_->ConnectMqttServer(local_id_);
	}
	bool CmdProcesserClient::SubscribeTopic()
	{
		return mqtt_wrapper_->SubscribeTopic(local_id_);
	}
	bool CmdProcesserClient::SendLocalCandidate(std::string tid, std::string type, std::string candidate)
	{
		std::string req;
		if (type.compare("offer") == 0)
		{
			req = MqttRequest::OfferRequest(clientMessageID(), local_id_, tid, candidate);
		}
		else if (type.compare("answer") == 0)
		{
			req = MqttRequest::AnswerRequest(clientMessageID(), local_id_, tid, candidate);
		}

		return mqtt_wrapper_->SendRequest(tid, req);
	}

	/**
	 * CmdObserver
	 */
	void CmdProcesserClient::OnServerConnectSuccess()
	{
		printf("connect mqtt server success!\n");
		SubscribeTopic();
	}
	void CmdProcesserClient::OnServerConnectFailure()
	{
		printf("connect mqtt server failure!\n");
	}
	void CmdProcesserClient::OnServerSubscribeSuccess()
	{
		printf("subscribe topic success!\n");
		
		std::string topic = kGlobalTopic;
		std::string req = MqttRequest::JoinRequest(clientMessageID(), local_id_);

		mqtt_wrapper_->SendRequest(topic, req);
	}
	void CmdProcesserClient::OnServerSubscribeFailure()
	{
		printf("subscribe topic failure!\n");
	}
	void CmdProcesserClient::OnServerMessageArrived(std::string topic, std::string message)
	{
		std::string method;
		std::string error;
		ReqValue	values;
		if (!MqttRequest::ParseRequestType(message, method, error))
		{
			printf("Error : Parse request error\n");
			return;
		}

		if (method.compare("result_code"))
		{
			return;
		}

		if (method.compare("publishers") == 0)
		{
			if (MqttRequest::ParsePublishersRequest(message, values, error))
			{
				OnParsePublishersSuccess(values["message_id"], values["publishers_id"]);
			}
			else
			{
				OnParseFailure(values["message_id"], values["publishers_id"], error);
			}
		}
		else if (method.compare("offer") == 0)
		{
			if (MqttRequest::ParseOfferRequest(message, values, error))
			{
				OnParseOfferOrAnswerSuccess(values["message_id"], "offer", values["from_id"], values["candidate"]);
			}
			else
			{
				OnParseFailure(values["message_id"], values["from_id"], error);
			}
		}
		else if (method.compare("answer") == 0)
		{
			if (MqttRequest::ParseAnswerRequest(message, values, error))
			{
				OnParseOfferOrAnswerSuccess(values["message_id"], "answer", values["from_id"], values["candidate"]);
			}
			else
			{
				OnParseFailure(values["message_id"], values["from_id"], error);
			}
		}
	}
	void CmdProcesserClient::OnParsePublishersSuccess(std::string message_id, std::string publishers_id)
	{
		publisher_list_.insert(std::pair<std::string, std::string>(publishers_id, publishers_id));

		std::string req = MqttRequest::BuildResponse(message_id, "ok");

		std::string topic = kGlobalTopic;
		mqtt_wrapper_->SendRequest(topic, req);
	}
	void CmdProcesserClient::OnParseFailure(std::string message_id, std::string from_id, std::string error)
	{
		std::string response = MqttRequest::BuildResponse(message_id, error);
		mqtt_wrapper_->SendRequest(from_id, response);
	}
	void CmdProcesserClient::OnParseOfferOrAnswerSuccess(std::string message_id, std::string type, std::string to_id, std::string candidate)
	{
		std::string req = MqttRequest::BuildResponse(message_id, "ok");
		mqtt_wrapper_->SendRequest(to_id, req);

		if (observer_)
		{
			observer_->OnReceiveRemoteCandidate(to_id, type, candidate);
		}
	}
}