package http

import (
	"fmt"
	"io"
	"os"
	"strings"
	"time"

	"github.com/gogf/gf/frame/g"
	"github.com/gogf/gf/net/ghttp"
	"github.com/gogf/gf/util/gconv"
)

func Get(url string) ([]byte, error) {
	return handle(g.Client().Timeout(30 * time.Second).Get(url))
}

func handle(response *ghttp.ClientResponse, err error) ([]byte, error) {
	if err != nil {
		return nil, err
	}
	defer response.Close()
	return response.ReadAll(), nil
}

type WriteCounter struct {
	Total uint64
	Length uint64
}

func (wc *WriteCounter) Write(p []byte) (int, error) {
	n := len(p)
	wc.Total += uint64(n)
	wc.PrintProgress()
	return n, nil
}

func (wc WriteCounter) PrintProgress() {
	// Clear the line by using a character return to go back to the start and remove
	// the remaining characters by filling it with spaces
	fmt.Printf("\r%s", strings.Repeat(" ", 35))

	// Return again and print current status of download
	// We use the humanize package to print the bytes in a meaningful way (e.g. 10 MB)
	fmt.Printf("\rDownloading... %s complete", fmt.Sprintf("%.2f", gconv.Float64(wc.Total)/gconv.Float64(wc.Length) * 100) + "%")
}




// 下载文件
func Download(file string, path string) error {
	g.Log().Async().Infof("Download File: %s", path)
	resp, err := g.Client().Get(file)

	if err != nil {
		return err
	}
	defer resp.Body.Close()

	// 创建一个文件用于保存
	out, err := os.Create(path)
	if err != nil {
		return err
	}



	defer out.Close()

	// 然后将响应流和文件流对接起来
	counter := &WriteCounter{Length: gconv.Uint64(resp.Header.Get("Content-Length"))}
	_, err = io.Copy(out, io.TeeReader(resp.Body, counter))

	if err != nil && err == io.EOF {
		return nil
	}
	if err != nil {
		return err
	}
	fmt.Print("\n")
	return nil
}
