package file

import (
	"archive/zip"
	"bytes"
	"encoding/json"
	"fmt"
	"image"
	"image/jpeg"
	"image/png"
	"os"
	"path"
	"regexp"
	"runtime/debug"
	"strings"

	"ppt_server/app/model"
	"ppt_server/library/http"
	"ppt_server/library/oss"

	"github.com/chai2010/webp"
	"github.com/gogf/gf/errors/gerror"
	"github.com/gogf/gf/frame/g"
	"github.com/gogf/gf/os/gtime"
	"github.com/xxjwxc/gowp/workpool"
)

// 解析URL返回基本信息
type ObtainFile struct {
	R         model.FileUploadRequest // 客户端请求参数
	ok        chan bool               // 上传课件响应成功
	err       chan error              // 上传课件失败Error
	fileSize  int64                   // 课件大小
	pageCount int                     // 课件图片数量
	parser    *parser                 // 课件对象
	n         notify                  // 事件通知
	packed    *model.Packed           // packed.json
}

const (
	parserFileLink     = "http://doc.offcncloud.com/"
	ossIntranetLink    = "xiaoyu-live.oss-cn-beijing-internal.aliyuncs.com"
	uploadFileLinkHost = "desktop.offcncloud.com"
)

var reFileNameCompile, _ = regexp.Compile(`p([\d]+)[.].*`)
var reTokenPattern = `<meta name="csrf-token" content="([^"]+)">`

func Upload(r model.FileUploadRequest, errChan chan error,
	ok chan bool) (file *ObtainFile, err error) {

	f, err := parseURL(r.Url)
	if err != nil {
		return nil, err
	}

	file = &ObtainFile{
		ok:     ok,
		err:    errChan,
		parser: f,
		R:      r,
		packed: &model.Packed{
			FileName:      f.filename(),
			Hash:          f.contentHash(),
			OssImagesPath: f.splitUploadPath() + "images/",
		},
		n: n,
	}
	return file, nil
}

//从web365获取上传文件信息
func (f *ObtainFile) fileInfo() error {
	fileBytes, err := http.Get(f.parser.splitFileInfoLink())
	if err != nil {
		return err
	}
	g.Log().Async().Infof("file info: %s", string(fileBytes))

	var fileInfo model.UploadFileInfo
	if err = json.Unmarshal(fileBytes, &fileInfo); err != nil {
		return err
	}

	if fileInfo.FileSize != 0 {
		f.fileSize = fileInfo.FileSize
	}
	switch {
	case fileInfo.SlideCount != 0:
		f.pageCount = fileInfo.SlideCount
	case fileInfo.PageCount != 0:
		f.pageCount = fileInfo.PageCount
	case fileInfo.SheetCount != 0:
		f.pageCount = fileInfo.SheetCount
	}

	return nil
}

// 获取图片宽和高 并将图片上传到OSS
func (f *ObtainFile) taskJob(file *zip.File) ([]interface{}, error) {
	fd, err := file.Open()
	if err != nil {
		return nil, err
	}
	defer fd.Close()
	var img image.Image
	fileSuffix := path.Ext(file.Name) //获取文件后缀
	switch fileSuffix {
	case ".png":
		img, err = png.Decode(fd)
	case ".jpeg":
		img, err = jpeg.Decode(fd)
	default:
		return nil, gerror.New("image ext is not found")
	}
	if err != nil {
		return nil, err
	}
	var buf bytes.Buffer
	if err = webp.Encode(&buf, img, &webp.Options{Lossless: true}); err != nil {
		return nil, err
	}
	replaceName := fmt.Sprintf("%05s", reFileNameCompile.ReplaceAllString(file.Name, "$1")) + ".webp"

	info := img.Bounds()
	imageInfo := []interface{}{replaceName, info.Max.X, info.Max.Y}

	return imageInfo, oss.Upload(f.parser.splitUploadPath()+"images/"+replaceName, bytes.NewReader(buf.Bytes()))
}

func (f *ObtainFile) task(file *zip.File) (err error) {
	defer func() {
		debug.FreeOSMemory()
	}()
	for i := 0; i < 3; i++ {
		if imageInfo, err := f.taskJob(file); err == nil {
			f.packed.ImageInfos = append(f.packed.ImageInfos, imageInfo)
			break
		}
	}
	return
}

func (f *ObtainFile) uploadPicture() error {

	reader, err := zip.OpenReader(f.parser.splitTempZipLink())
	if err != nil {
		return err
	}
	defer reader.Close()

	pool := workpool.New(100)

	for _, file := range reader.File {
		tempFile := file
		pool.Do(func() error {
			return f.task(tempFile)
		})
	}
	return pool.Wait()
}

// 上传packed.json
func (f *ObtainFile) uploadPacked() error {
	for i := 1; i <= f.pageCount; i++ {
		f.packed.ImagesName = append(f.packed.ImagesName, fmt.Sprintf("%05d", i)+".png")
	}
	packedBytes, err := json.Marshal(f.packed)
	if err != nil {
		return err
	}
	return oss.Upload(f.parser.splitUploadPath()+"packed.json", bytes.NewReader(packedBytes))
}

// 拼接下载zip包地址
func (f *ObtainFile) splitDownloadLink() string {
	return fmt.Sprintf("%s?info=1&words=%v&ssl=1&furl=%s/%s",
		parserFileLink, f.pageCount, g.Cfg().GetString("oss.url"),
		f.parser.dataPath())
}

func (f *ObtainFile) deferWorker(err error) {
	// 清理ZIP包
	_ = os.Remove(f.parser.splitTempZipLink())
	// 通知
	if err != nil {
		g.Log().Async().Error(err)
		f.err <- err
		return
	}
	f.ok <- true
}

// 进行解析上传操作
func (f *ObtainFile) Worker(err error) {
	defer func() {
		f.deferWorker(err)
	}()
	if err != nil {
		return
	}
	// 解析课件获取课件信息
	if err = f.fileInfo(); err != nil {
		return
	}
	g.Log().Async().Infof("[1/4] file: %s file info complete", f.parser.filename())

	// 下载ZIP包
	if err = http.Download(f.splitDownloadLink(), f.parser.splitTempZipLink()); err != nil {
		return
	}
	g.Log().Async().Infof("[2/4] file: %s zip donwload complete", f.parser.filename())

	// 解析ZIP包并上传图片到OSS
	if err = f.uploadPicture(); err != nil {
		return
	}
	g.Log().Async().Infof("[3/4] file: %s picture upload oss complete", f.parser.filename())

	// 上传packed.json到oss
	if err = f.uploadPacked(); err != nil {
		return
	}
	g.Log().Async().Infof("[4/4] file: %s upload complete!", f.parser.filename())
}

// 上传失败通知
func (f *ObtainFile) Fail() {
	if err := f.n.notice("/web/room_files_error", g.Map{
		"code":     400,
		"uuid":     f.R.Uuid,
		"room_num": f.R.RoomNum,
		"path":     f.parser.dataPath(),
	}); err != nil {
		g.Log().Async().Error(err)
	}
	g.Log().Async().Infof("Fail Notify Success!")
}

// 上传成功通知
func (f *ObtainFile) Success(roomID int) {

	if err := f.n.notice("/web/room_files_add", g.Map{
		"name":           f.R.Name,
		"file_name_hash": f.parser.nameHash(),
		"hash":           f.parser.contentHash(),
		"path":           f.parser.dataPath(),
		"doc_type":       f.parser.docType(),
		"type":           f.parser.mineType(),
		"files_size":     f.fileSize,
		"page_count":     f.pageCount,
		"link":           strings.Replace(f.parser.url, ossIntranetLink, uploadFileLinkHost, -1),
		"private":        "1",
		"time":           gtime.Datetime(),
		"nickname":       f.R.NickName,
		"uuid":           f.R.Uuid,
		"room_id":        roomID,
		"is_courseware":  1,
	}); err != nil {
		g.Log().Async().Error(err)
	}
	g.Log().Async().Infof("Success Notify Success!")
}
