package file

import (
	"encoding/json"
	"os"

	"ppt_server/app/model"
	"ppt_server/library/http"

	"github.com/gogf/gf/errors/gerror"
	"github.com/gogf/gf/frame/g"
	"github.com/gogf/gf/text/gregex"
)

type Notify struct {
	name     string
	password string
	url      string
	client   *http.SessionHttpClient
}

func NewNotify() *Notify {
	return &Notify{
		url:      g.Cfg().GetString("notify.url"),
		name:     os.Getenv("live_name"),
		password: os.Getenv("live_pwd"),
		client:   http.NewSessionHttpClient(),
	}
}

//回调验权
func (n *Notify) verify(url string, data g.Map) error {

	res, err := n.client.Post(url, data)
	if err != nil {
		return err
	}
	
	var notifyResponse model.NotifyResponse
	if err = json.Unmarshal(res, &notifyResponse); err != nil {
		return err
	}

	if notifyResponse.Code != 200 {
		return gerror.New(notifyResponse.Msg)
	}
	return nil
}

// 登录
func (n *Notify) auth(data g.Map) error {

	if err := n.verify(n.url+"/web/login", g.Map{
		"name":     n.name,
		"password": n.password,
	}); err != nil {
		return err
	}
	// 获取Token
	bufferBytes, err := n.client.Get(n.url + "/web/admin")
	if err != nil {
		return err
	}

	buf, _ := gregex.Match(`<meta name="csrf-token" content="([^"]+)">`, bufferBytes)
	data["_token"] = string(buf[1])

	return nil
}

// 发送通知
func (n *Notify) notify(path string, data g.Map) error {
	var err error
	if err = n.auth(data); err != nil {
		return err
	}
	if err = n.verify(n.url+path, data); err != nil {
		return err
	}
	g.Log().Async().Infof("Notify Success!")
	return nil
}

// 上传课件失败通知
func (n *Notify) Fail(object *ObtainFile) {
	if err := n.notify("/web/room_files_error", g.Map{
		"code":     400,
		"uuid":     object.r.Uuid,
		"room_num": object.r.RoomNum,
		"path":     object.parser.dataPath,
	}); err != nil {
		g.Log().Async().Error(err)
	}
}

// 上传成功通知
func (n *Notify) Success(res g.Map) {
	if err := n.notify("/web/room_files_add", res); err != nil {
		g.Log().Async().Error(err)
	}
}
