package file

import (
	"archive/zip"
	"bytes"
	"encoding/json"
	"fmt"
	"image"
	"image/jpeg"
	"image/png"
	"io/ioutil"
	"os"
	"path"
	"regexp"
	"strings"

	"ppt_server/app/model"
	"ppt_server/library/http"
	"ppt_server/library/oss"

	"github.com/gogf/gf/container/gset"
	"github.com/gogf/gf/errors/gerror"
	"github.com/gogf/gf/frame/g"
	"github.com/gogf/gf/os/gtime"
	"github.com/xxjwxc/gowp/workpool"
)

// 解析URL返回基本信息
type ObtainFile struct {
	r         *model.FileUploadRequest // 客户端请求参数
	ok        chan bool                // 上传课件响应成功
	err       chan error               // 上传课件失败Error
	fileSize  int64                    // 课件大小
	pageCount int                      // 课件图片数量
	parser    *parser                  // 课件对象
	n         *notify                  // 事件通知
	packed    *model.Packed            // packed.json
}

const (
	ParserFileLink     = "http://doc.offcncloud.com/"
	OssIntranetLink    = "xiaoyu-live.oss-cn-beijing-internal.aliyuncs.com"
	UploadFileLinkHost = "desktop.offcncloud.com"
)

var reFileNameCompile, _ = regexp.Compile(`p([\d]+)[.].*`)


func Upload(r *model.FileUploadRequest, errChan chan error,
	ok chan bool) (file *ObtainFile, err error) {

	f, err := parseURL(r.Url)
	if err != nil {
		return nil, err
	}

	file = &ObtainFile{
		ok:     ok,
		err:    errChan,
		parser: f,
		r:      r,
		packed: &model.Packed{
			FileName:      f.filename(),
			Hash:          f.contentHash(),
			ImagesName:    gset.NewSet(),
			OssImagesPath: f.splitUploadPath() + "images/",
		},
		n: newNotify(),
	}
	return file, nil
}

//从web365获取上传文件信息
func (f *ObtainFile) fileInfo() error {
	fileBytes, err := http.Get(f.parser.splitFileInfoLink())
	if err != nil {
		return err
	}
	var fileInfo model.UploadFileInfo
	if err = json.Unmarshal(fileBytes, &fileInfo); err != nil {
		return err
	}

	if fileInfo.FileSize != 0 {
		f.fileSize = fileInfo.FileSize
	}

	switch {
	case fileInfo.SlideCount != 0:
		f.pageCount = fileInfo.SlideCount
	case fileInfo.PageCount != 0:
		f.pageCount = fileInfo.PageCount
	case fileInfo.SheetCount != 0:
		f.pageCount = fileInfo.SheetCount
	}

	return nil
}

// 获取图片宽和高 并将图片上传到OSS
func (f *ObtainFile) taskJob(file *zip.File) ([]interface{}, error) {
	fd, err := file.Open()
	if err != nil {
		return nil, err
	}

	defer fd.Close()

	pictureBytes, _ := ioutil.ReadAll(fd)
	// 获取图片信息reader
	imageReader := ioutil.NopCloser(bytes.NewBuffer(pictureBytes))
	// 上传图片reader
	fileReader := ioutil.NopCloser(bytes.NewBuffer(pictureBytes))

	var img image.Image
	fileSuffix := path.Ext(file.Name) //获取文件后缀
	switch fileSuffix {
	case ".png":
		img, err = png.Decode(imageReader)
	case ".jpeg":
		img, err = jpeg.Decode(imageReader)
	default:
		return nil, gerror.New("image ext is not found")
	}
	if err != nil {
		return nil, err
	}
	replaceName := fmt.Sprintf("%05s", reFileNameCompile.ReplaceAllString(file.Name, "$1")) + fileSuffix

	info := img.Bounds()
	imageInfo := []interface{}{replaceName, info.Max.X, info.Max.Y}

	g.Log().Async().Infof("Upload picture file: %s Success", replaceName)

	return imageInfo, oss.Upload(f.parser.splitUploadPath()+"images/"+replaceName, fileReader)
}

func (f *ObtainFile) task(file *zip.File) error {
	var (
		err       error
		imageInfo []interface{}
	)
	for i := 0; i < 3; i++ {
		if imageInfo, err = f.taskJob(file); err == nil {
			f.packed.ImageInfos = append(f.packed.ImageInfos, imageInfo)
			f.packed.ImagesName.AddIfNotExist(imageInfo[0])
			break
		}
	}
	return err
}

func (f *ObtainFile) uploadPicture() error {

	reader, err := zip.OpenReader(f.parser.splitTempZipLink())
	if err != nil {
		return err
	}
	defer reader.Close()

	pool := workpool.New(100)
	for _, file := range reader.File {
		pool.DoWait(func() error {
			return f.task(file)
		})
	}
	return pool.Wait()
}

// 上传packed.json
func (f *ObtainFile) uploadPacked() error {
	packedBytes, _ := json.Marshal(f.packed)
	return oss.Upload(f.parser.splitUploadPath()+"packed.json", bytes.NewReader(packedBytes))
}

// 进行解析上传操作
func (f *ObtainFile) Worker(err error) {
	if err != nil {
		return
	}

	// 解析课件获取课件信息
	if err = f.fileInfo(); err != nil {
		g.Log().Async().Error(err)
		f.err <- err
		return
	}
	g.Log().Async().Infof("file: %s file info complete", f.parser.name)

	downloadZipLink := fmt.Sprintf("%s?info=1&words=%v&ssl=1&furl=%s/%s",
		ParserFileLink, f.pageCount, g.Cfg().GetString("oss.url"),
		f.parser.dataPath())

	// 下载ZIP包
	if err = http.Download(downloadZipLink, f.parser.splitTempZipLink()); err != nil {
		g.Log().Async().Error(err)
		f.err <- err
		return
	}
	// 清理ZIP包
	defer os.Remove(f.parser.splitTempZipLink())
	g.Log().Async().Infof("file: %s Zip Donwload Complete", f.parser.filename())

	// 解析ZIP包并上传图片到OSS
	if err = f.uploadPicture(); err != nil {
		g.Log().Async().Error(err)
		f.err <- err
		return
	}
	// 上传packed.json到oss
	if err = f.uploadPacked(); err != nil {
		g.Log().Async().Error(err)
		f.err <- err
		return
	}
	g.Log().Async().Infof("filename: %s upload success!", f.parser.name)
	f.ok <- true
}

func (f *ObtainFile) FileName() string {
	return f.parser.filename()
}

// 上传失败通知
func (f *ObtainFile) Fail() {
	if err := f.n.notice("/web/room_files_error", g.Map{
		"code":     400,
		"uuid":     f.r.Uuid,
		"room_num": f.r.RoomNum,
		"path":     f.parser.dataPath(),
	}); err != nil {
		g.Log().Async().Error(err)
	}
}

// 上传成功通知
func (f *ObtainFile) Success(roomID int) {
	m := g.Map{
		"name":           f.r.Name,
		"file_name_hash": f.parser.nameHash(),
		"hash":           f.parser.contentHash(),
		"path":           f.parser.dataPath(),
		"doc_type":       f.parser.docType(),
		"type":           f.parser.mineType(),
		"files_size":     f.fileSize,
		"page_count":     f.pageCount,
		"link":           strings.Replace(f.parser.url, OssIntranetLink, UploadFileLinkHost, -1),
		"private":        "1",
		"time":           gtime.Datetime(),
		"nickname":       f.r.Nickanem,
		"uuid":           f.r.Uuid,
		"room_id":        roomID,
		"is_courseware" : 1,
	}

	if err := f.n.notice("/web/room_files_add", m); err != nil {
		g.Log().Async().Error(err)
	}
}
