package im_user

import (
	"context"
	"errors"
	"im-microservice/db"
	"im-microservice/helper"
	"im-microservice/pb"
)

type ImUserServer struct {
	pb.UnimplementedImUserServer
}

type ImUser struct {
	Name     string `validate:"required"`
	Mobile   string `validate:"required,numeric,len=11"`
	Code     string `validate:"required"`
	Platfrom string `validate:"required"`
}

type ImUserSet struct {
	Appid string `validate:"required"`
	Mute  bool   `validate:"required"`
}

type ImUserQ struct {
	Appid string `validate:"required"`
}

//注册
func (u *ImUserServer) ImUserRegister(ctx context.Context, in *pb.ImUserRequest) (reply *pb.ImUserReply, err error) {
	imUser := &ImUser{
		Name:   in.GetName(),
		Mobile: in.GetMobile(),
		Code:   in.GetCode(),
	}

	// 参数验证是否合法
	if err = helper.Valiator(imUser); err != nil {
		return
	}

	// 从redis获取短信验证码
	cache_code, err := helper.GetCode(imUser.Mobile)
	if err != nil {
		return
	}
	//对比验证码
	if cache_code != imUser.Code {
		err = errors.New("验证码不匹配")
		return
	}
	reply = &pb.ImUserReply{}
	// 是否已经注册过, 如果已经注册返回用户信息
	if c_user, _ := GetImUserByMobile(imUser.Mobile); c_user.Appid != "" {
		reply.Appid = c_user.Appid

		return
	}
	// 企业用户注册失败
	c_user, err := helper.CreateImUser(imUser.Name, imUser.Mobile)
	if err != nil {
		return
	}
	reply.Appid = c_user.Appid
	return
}

//用户登录
func (u *ImUserServer) ImUserLogin(ctx context.Context, in *pb.ImUserRequest) (reply *pb.ImUserReply, err error) {
	imUser := &ImUser{
		Name:     in.GetName(),
		Mobile:   in.GetMobile(),
		Code:     in.GetCode(),
		Platfrom: in.GetPlatfrom(),
	}

	// 参数验证是否合法
	if err = helper.Valiator(imUser); err != nil {
		return
	}

	// 从redis获取短信验证码
	cache_code, err := helper.GetCode(imUser.Mobile)
	if err != nil {
		return
	}
	//对比验证码
	if cache_code != imUser.Code {
		err = errors.New("验证码不匹配")
		return
	}
	reply = &pb.ImUserReply{}
	c_user, _ := GetImUserByMobile(imUser.Mobile)
	if c_user.Appid == "" {
		err = errors.New("用户不存在")
		return
	}
	//更新用户登录时间
	if res, _ := SaveUpdatetime(imUser.Mobile, imUser.Platfrom); res != 1 {
		err = errors.New("用户登录时间更新失败")
		return
	}
	reply.Appid = c_user.Appid
	return
}

//用户退出
func (u *ImUserServer) ImUserLoginOut(ctx context.Context, in *pb.ImUserOutRequest) (reply *pb.ImUserSetReply, err error) {
	imUserOut := &ImUserQ{
		Appid: in.GetAppid(),
	}
	if res, _ := SaveLoginOut(imUserOut.Appid); res != 1 {
		err = errors.New("用户更改登录状态信息失败")
		return
	}
	reply = &pb.ImUserSetReply{}
	return
}

//用户禁言
func (u *ImUserServer) ImUserForbiddenWords(ctx context.Context, in *pb.ImUserSetRequest) (reply *pb.ImUserSetReply, err error) {
	imUserSet := &ImUserSet{
		Appid: in.GetAppid(),
		Mute:  in.GetMute(),
	}
	if res, _ := SetImUserMute(imUserSet.Appid, "mute_estoppel", imUserSet.Mute); res != 1 {
		err = errors.New("用户禁言失败")
		return
	}

	reply = &pb.ImUserSetReply{}
	return
}

//用户禁音视频
func (u *ImUserServer) ImUserForbiddenAV(ctx context.Context, in *pb.ImUserSetRequest) (reply *pb.ImUserSetReply, err error) {
	imUserSet := &ImUserSet{
		Appid: in.GetAppid(),
		Mute:  in.GetMute(),
	}
	if res, _ := SetImUserMute(imUserSet.Appid, "mute_audio_video", imUserSet.Mute); res != 1 {
		err = errors.New("用户禁言失败")
		return
	}

	reply = &pb.ImUserSetReply{}
	return
}

//获取用户信息
func (u *ImUserServer) ImUserOne(ctx context.Context, in *pb.ImUserOneRequest) (reply *pb.ImUserOneReply, err error) {
	imUserId := &ImUserQ{
		Appid: in.GetAppid(),
	}
	// 参数验证是否合法
	if err = helper.Valiator(imUserId); err != nil {
		return
	}
	reply = &pb.ImUserOneReply{}
	c_user, err := GetImUserByAppid(imUserId.Appid)
	if err != nil {
		return
	}
	reply = getImUser(c_user)
	return
}

func getImUser(imuser db.ImUser) *pb.ImUserOneReply {
	return &pb.ImUserOneReply{
		Id:             int64(imuser.Id),
		Appid:          imuser.Appid,
		Name:           imuser.Name,
		Mobile:         imuser.Mobile,
		Disable:        int64(imuser.Disable),
		MuteEstoppel:   int64(imuser.MuteEstoppel),
		MuteAudioVideo: int64(imuser.MuteAudioVideo),
		Ext:            imuser.Ext,
		Createtime:     imuser.Createtime,
		Updatetime:     imuser.Updatetime,
		LoginStatus:    int64(imuser.LoginStatus),
		Platfrom:       imuser.Platfrom,
		Edition:        imuser.Edition,
	}
}

//获取用户列表
func (u *ImUserServer) ImUserList(ctx context.Context, in *pb.ImUserListRequest) (reply *pb.ImUserListReply, err error) {
	results, err := GetImUserList(in)
	if err != nil {
		return
	}

	reply = &pb.ImUserListReply{Paginate: &pb.Page{}}

	im_user_list := results["list"].([]db.ImUser)
	for _, v := range im_user_list {
		reply.List = append(reply.List, getImUser(v))
	}

	reply.Paginate.PageSize = int64(results["page_size"].(int))
	reply.Paginate.PagetNo = results["page"].(int64)
	reply.Paginate.TotalCount = results["total_count"].(int64)
	reply.Paginate.TotalPage = results["total_page"].(int64)
	return
}
