package main

import (
	"crypto/sha1"
	"encoding/hex"
	"errors"
	"fmt"
	"im-microservice/sevice/im_user"
	"log"
	"net"
	"regexp"
	"strconv"
	"time"

	"im-microservice/helper"
	"im-microservice/pb"
	"im-microservice/sevice/im_chat_room"
	ic "im-microservice/sevice/im_configure"

	"golang.org/x/net/context"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/metadata"
)

const (
	port = ":50051"
)

func checksum(req map[string]string) error {

	err, AppSecret := helper.GetSecretByKey(req["Appkey"])
	if err != nil {
		return errors.New("appkey不存在")
	}

	server_time := time.Now().Unix()
	client_time, err := strconv.Atoi(req["Curtime"])
	if err != nil {
		return err
	}
	if server_time > int64(client_time) {
		return errors.New("当前客户端时间不能小于服务端时间")
	}
	sha11 := sha1.New()
	sha11.Write([]byte(AppSecret + req["Nonce"] + req["Curtime"]))
	sevice_checksum := hex.EncodeToString(sha11.Sum([]byte(nil)))
	if sevice_checksum != req["Checksum"] {
		return errors.New("加密串不正确")
	}
	return nil
}

func auth(ctx context.Context, req interface{}, info *grpc.UnaryServerInfo, handler grpc.UnaryHandler) (interface{}, error) {

	_, ok := metadata.FromIncomingContext(ctx)
	if !ok {
		return nil, grpc.Errorf(codes.Unauthenticated, "无Token认证信息")
	}

	request_map := make(map[string]string)
	s := fmt.Sprintf("%v", req)

	key_r, _ := regexp.Compile(`Appkey:"(.*?)"`)
	nonce_r, _ := regexp.Compile(`Nonce:"(.*?)"`)
	c_r, _ := regexp.Compile(`Curtime:"(.*?)"`)
	cs_r, _ := regexp.Compile(`Checksum:"(.*?)"`)

	appkey := key_r.FindAllStringSubmatch(s, -1)
	nonce := nonce_r.FindAllStringSubmatch(s, -1)
	curtime := c_r.FindAllStringSubmatch(s, -1)
	check_sum := cs_r.FindAllStringSubmatch(s, -1)

	if len(appkey) < 1 {
		return nil, errors.New("缺少appkey")
	}
	if len(nonce) < 1 {
		return nil, errors.New("缺少nonce")
	}
	if len(curtime) < 1 {
		return nil, errors.New("缺少curtime")
	}
	if len(check_sum) < 1 {
		return nil, errors.New("缺少checksum")
	}

	request_map["Appkey"] = appkey[0][1]
	request_map["Nonce"] = nonce[0][1]
	request_map["Curtime"] = curtime[0][1]
	request_map["Checksum"] = check_sum[0][1]

	if err := checksum(request_map); err != nil {
		return nil, err
	}

	return handler(ctx, req)
}

func main() {

	lis, err := net.Listen("tcp", port)
	if err != nil {
		log.Fatalf("failed to listen: %v", err)
	}

	//var opts []grpc.ServerOption
	//opts = append(opts, grpc.UnaryInterceptor(auth))
	//s := grpc.NewServer(opts...)
	s := grpc.NewServer()
	pb.RegisterConfigureSeviceServer(s, &ic.ConfigureSevice{})
	pb.RegisterChatRoomServiceServer(s, &im_chat_room.ImChatRoomService{})
	pb.RegisterImUserServer(s, &im_user.ImUserServer{})
	if err := s.Serve(lis); err != nil {
		log.Fatalf("failed to serve: %v", err)
	}

}
