package im_chat_room

import (
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"im-microservice/db"
	"im-microservice/helper"
	"im-microservice/pb"
	"im-microservice/validator_struct"

	"github.com/astaxie/beego/orm"
)

const (
	chat_table_name   = "im_chat_room"
	chat_room_id      = "id"
	chat_creator      = "creator"
	chat_room         = "name"
	chat_announcement = "announcement"
	chat_status       = "status"
	chat_create_time  = "createtime"
)

func AddChatRoom(addStruct *validator_struct.ChatRoomAddStruct) (int64, error) {
	var chat db.ImChatRoom
	now := helper.GetNowTime()
	chat.Name = addStruct.Name
	chat.Announcement = addStruct.Announcement
	chat.Creator = addStruct.Creator
	chat.Status = 1
	chat.Createtime = now
	chat.Updatetime = now
	room_id, err := db.MysqlClient.Insert(&chat)
	return room_id, status.Error(codes.Internal, err.Error())
}

func UpdateChatRoom(update *validator_struct.ChatRoomUpdate) error {
	orm_params := make(orm.Params)
	if update.Status != 0 {
		orm_params[chat_status] = update.Status
	}
	if update.Announcement != "" {
		orm_params[chat_announcement] = update.Announcement
	}
	if update.Name != "" {
		orm_params[chat_room] = update.Name
	}

	if len(orm_params) == 0 {
		return status.Error(codes.InvalidArgument, "修改房间的参数为空")
	}
	orm_params["updatetime"] = helper.GetNowTime()
	_, err := db.MysqlClient.QueryTable(chat_table_name).
		Filter(chat_room_id, update.RoomId).Update(orm_params)

	return err
}

func DeleteChatRoom(room_id int64) error {
	_, err := db.MysqlClient.QueryTable(chat_table_name).
		Filter(chat_room_id, room_id).Delete()
	if err != nil {
		return status.Error(codes.Internal, err.Error())
	}
	return nil

}

func ChatRoomInfo(info *validator_struct.ChatRoomInfo) (res db.ImChatRoom, err error) {
	err = db.MysqlClient.QueryTable(chat_table_name).Filter(chat_room_id, info.RoomId).One(&res)
	if err != nil {
		err = status.Error(codes.Internal, err.Error())
	}
	return
}

func GetChatRooms(request *pb.ChatRoomAllRequest) (results map[string]interface{}, err error) {
	var (
		announcement = request.GetAnnouncement()
		name         = request.GetName()
		req_status   = request.GetStatus()
		creator      = request.GetCreator()
		createtime   = request.GetCreatetime()
		page         = request.GetPage()
	)
	if page == 0 {
		page = 1
	}
	count_db := db.MysqlClient.QueryTable(chat_table_name)
	if announcement != "" {
		count_db = count_db.Filter(chat_announcement+"__icontains", announcement)
	}
	if name != "" {
		count_db = count_db.Filter(chat_room+"__icontains", name)
	}
	if req_status != 0 {
		count_db = count_db.Filter(chat_status, req_status)
	}
	if creator != "" {
		count_db = count_db.Filter(chat_creator, creator)
	}
	if createtime != "" {
		count_db = count_db.Filter(chat_create_time+"__gte", createtime)
	}
	ctn, err := count_db.Count()
	if err != nil {
		return
	}
	if ctn == 0 {
		return
	}
	// 获取查询条件聊天室的总数
	offset, total_page := helper.Paginate(ctn, 20, page)

	// 查询数据
	var chat_rooms []db.ImChatRoom
	if _, err = count_db.Limit(20, offset).All(&chat_rooms); err != nil {
		return
	}
	results = make(map[string]interface{})
	results["list"] = chat_rooms
	results["total_page"] = total_page
	results["total_count"] = ctn
	results["page"] = page
	results["page_size"] = 20
	return
}
