package main

import (
	"crypto/sha1"
	"encoding/hex"
	"fmt"
	"log"
	"net"
	"regexp"
	"strconv"
	"sync"
	"time"

	"im-microservice/helper"
	"im-microservice/pb"
	"im-microservice/sevice/im_chat_room"
	ic "im-microservice/sevice/im_configure"
	"im-microservice/sevice/im_user"
	iur "im-microservice/sevice/im_user_relationship"

	"golang.org/x/net/context"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/reflection"
	"google.golang.org/grpc/status"
)

type Health interface {
	// Check returns if server is healthy or not
	Check(c context.Context) (bool, error)
}

// 健康检测
type Server struct {
	mu        sync.Mutex
	statusMap map[string]pb.HealthCheckResponse_ServingStatus
}

func NewServer() *Server {
	return &Server{
		statusMap: make(map[string]pb.HealthCheckResponse_ServingStatus),
	}
}
func permissionError(msg string) error {
	return status.Error(codes.InvalidArgument, msg)
}

func (s *Server) Check(ctx context.Context, in *pb.HealthCheckRequest) (*pb.HealthCheckResponse, error) {
	s.mu.Lock()
	defer s.mu.Unlock()
	if in.Service == "" {
		return &pb.HealthCheckResponse{Status: pb.HealthCheckResponse_SERVING}, nil
	}
	if status, ok := s.statusMap[in.Service]; ok {
		return &pb.HealthCheckResponse{Status: status}, nil
	}
	return nil, status.Error(codes.NotFound, "unkonw service")
}

const (
	port        = ":50051"
	health_path = "/pb.Health/Check"
)

func checksum(req map[string]string) error {

	err, AppSecret := helper.GetSecretByKey(req["Appkey"])
	if err != nil {
		return status.Error(codes.NotFound, "appkey不存在")
	}

	server_time := time.Now().Unix()
	client_time, err := strconv.Atoi(req["Curtime"])
	if err != nil {
		return status.Error(codes.Aborted, err.Error())
	}
	if server_time > int64(client_time) {
		return permissionError("当前客户端时间不能小于服务端时间")
	}

	sha11 := sha1.New()
	sha11.Write([]byte(AppSecret + req["Nonce"] + req["Curtime"]))
	sevice_checksum := hex.EncodeToString(sha11.Sum([]byte(nil)))

	if sevice_checksum != req["Checksum"] {
		return permissionError("加密串不正确")
	}
	return nil
}

func auth(ctx context.Context, req interface{}, info *grpc.UnaryServerInfo, handler grpc.UnaryHandler) (interface{}, error) {

	if info.FullMethod != health_path {
		request_map := make(map[string]string)
		s := fmt.Sprintf("%v", req)

		key_r, _ := regexp.Compile(`Appkey:"(.*?)"`)
		nonce_r, _ := regexp.Compile(`Nonce:"(.*?)"`)
		c_r, _ := regexp.Compile(`Curtime:"(.*?)"`)
		cs_r, _ := regexp.Compile(`Checksum:"(.*?)"`)

		appkey := key_r.FindAllStringSubmatch(s, -1)
		nonce := nonce_r.FindAllStringSubmatch(s, -1)
		curtime := c_r.FindAllStringSubmatch(s, -1)
		check_sum := cs_r.FindAllStringSubmatch(s, -1)

		if len(appkey) < 1 {
			return nil, permissionError("缺少appkey")
		}
		if len(nonce) < 1 {
			return nil, permissionError("缺少nonce")
		}
		if len(curtime) < 1 {
			return nil, permissionError("缺少curtime")
		}
		if len(check_sum) < 1 {
			return nil, permissionError("缺少checksum")
		}
		request_map["Appkey"] = appkey[0][1]
		request_map["Nonce"] = nonce[0][1]
		request_map["Curtime"] = curtime[0][1]
		request_map["Checksum"] = check_sum[0][1]
		if err := checksum(request_map); err != nil {
			return nil, err
		}
	}
	return handler(ctx, req)
}

func main() {

	lis, err := net.Listen("tcp", port)
	if err != nil {
		log.Fatalf("failed to listen: %v", err)
	}

	var opts []grpc.ServerOption
	opts = append(opts, grpc.UnaryInterceptor(auth))
	s := grpc.NewServer(opts...)

	srv := NewServer()
	pb.RegisterHealthServer(s, srv)
	pb.RegisterConfigureSeviceServer(s, &ic.ConfigureSevice{})
	pb.RegisterChatRoomServiceServer(s, &im_chat_room.ImChatRoomService{})
	pb.RegisterImUserServer(s, &im_user.ImUserServer{})
	pb.RegisterUserRelationshipServiceServer(s, &iur.UserRelationshipService{})
	reflection.Register(s)
	
	log.Println("gRPC server is running on " + port + " port.")
	if err := s.Serve(lis); err != nil {
		log.Fatalf("failed to serve: %v", err)
	}

}
