package middleware

import (
	"context"
	"crypto/sha1"
	"encoding/hex"
	"fmt"
	"regexp"
	"strconv"
	"time"

	"im-microservice/helper"

	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

const (
	health_path = "/pb.Health/Check"
)

func permissionError(msg string) error {
	return status.Error(codes.InvalidArgument, msg)
}

func checksum(req map[string]string) error {

	err, AppSecret := helper.GetSecretByKey(req["Appkey"])
	if err != nil {
		return status.Error(codes.NotFound, "appkey不存在")
	}

	server_time := time.Now().Unix()
	client_time, err := strconv.Atoi(req["Curtime"])
	if err != nil {
		return status.Error(codes.Aborted, err.Error())
	}
	if server_time > int64(client_time) {
		return permissionError("当前客户端时间不能小于服务端时间")
	}

	sha11 := sha1.New()
	sha11.Write([]byte(AppSecret + req["Nonce"] + req["Curtime"]))
	sevice_checksum := hex.EncodeToString(sha11.Sum([]byte(nil)))

	if sevice_checksum != req["Checksum"] {
		return permissionError("加密串不正确")
	}
	return nil
}

func Auth(ctx context.Context, req interface{}, info *grpc.UnaryServerInfo,
	handler grpc.UnaryHandler) (interface{}, error) {

	if info.FullMethod != health_path {

		request_map := make(map[string]string)
		s := fmt.Sprintf("%v", req)

		key_r, _ := regexp.Compile(`Appkey:"(.*?)"`)
		nonce_r, _ := regexp.Compile(`Nonce:"(.*?)"`)
		c_r, _ := regexp.Compile(`Curtime:"(.*?)"`)
		cs_r, _ := regexp.Compile(`Checksum:"(.*?)"`)

		appkey := key_r.FindAllStringSubmatch(s, -1)
		nonce := nonce_r.FindAllStringSubmatch(s, -1)
		curtime := c_r.FindAllStringSubmatch(s, -1)
		check_sum := cs_r.FindAllStringSubmatch(s, -1)


		request_map["Appkey"] = appkey[0][1]
		request_map["Nonce"] = nonce[0][1]
		request_map["Curtime"] = curtime[0][1]
		request_map["Checksum"] = check_sum[0][1]
		if err := checksum(request_map); err != nil {
			return nil, err
		}

	}
	return handler(ctx, req)
}
